package com.jxjy.system.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.jxjy.system.bean.auth.UserContext;
import com.jxjy.system.mapper.MessageRecordMapper;
import com.jxjy.system.service.MessageRecordService;
import com.jxjy.system.service.RecruitService;
import com.jxjy.system.util.auth.UserUtils;
import com.jxjy.system.xsnowflake.SnowFlakeFactory;
import data.system.MessageRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

/**
 *
 * @return: 消息管理
 * @Author: wml
 * @Date 2025/5/9 10:22
 */
@Service
@Transactional
public class MessageRecordServiceImpl extends ServiceImpl<MessageRecordMapper, MessageRecord> implements MessageRecordService {

    @Autowired
    SnowFlakeFactory snowFlakeFactory;

    @Autowired
    RecruitService recruitService;

    /**
     * @description: 获取企业列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.messageRecord.MessageRecord>
     **/
    @Override
    public List<MessageRecord> list(int pageNum, int pageSize, MessageRecord messageRecord) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        //创建查询条件
        LambdaQueryWrapper<MessageRecord> qw = new LambdaQueryWrapper<>();
        qw.eq(MessageRecord::getDel,false)
                .eq(MessageRecord::getPersonnelId, UserUtils.getUserContext().getUserId())
                .eq(StringUtils.isNotBlank(messageRecord.getBaseCode()),MessageRecord::getBaseCode, messageRecord.getBaseCode())
                .orderByDesc(MessageRecord::getCreateTime);
        List<MessageRecord> messageRecordList = super.list(qw);

        List<String> ids = messageRecordList.stream().map(MessageRecord::getId).collect(Collectors.toList());

        if(ids.size()>0){
            //标记为已读
            LambdaUpdateWrapper<MessageRecord> uw = new LambdaUpdateWrapper<>();
            uw.in(MessageRecord::getId, ids);
            uw.set(MessageRecord::getIsRead,true);
            super.update(uw);
        }

        messageRecordList.stream().forEach(returnMessageRecord -> {
            returnMessageRecord.setRecruit(recruitService.getDetails(returnMessageRecord.getRecruitId()));
        });

        //返回查询结果
        return messageRecordList;
    }


    /**
     * @description: 新增企业
     * @author: wangmenglong
     * @date; 2023/12/11 16:51
     * @param: [messageRecord]
     * @return: boolean
     **/
    @Override
    public boolean create(MessageRecord messageRecord) {

        messageRecord.setId(snowFlakeFactory.nextId());
        messageRecord.setDel(false);
        messageRecord.setCreateTime(DateUtil.now());
        messageRecord.setIsRead(false);

        return super.save(messageRecord);
    }

    /**
     * @description: 编辑企业
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [messageRecord]
     * @return: boolean
     **/
    @Override
    public boolean update(MessageRecord messageRecord) {
        messageRecord.setUpdateTime(DateUtil.now());
        //判断是否传入id
        if(StringUtils.isBlank(messageRecord.getId())){
            return false;
        }
        return super.saveOrUpdate(messageRecord);
    }

    /**
     * @description: 删除企业
     * @author: wangmenglong
     * @date; 2023/12/11 17:06
     * @param: [id]
     * @return: boolean
     **/
    @Override
    public boolean del(String id) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        LambdaUpdateWrapper<MessageRecord> qw = new LambdaUpdateWrapper<>();
        qw.eq(MessageRecord::getId,id)
                .set(MessageRecord::getDel,true)
                .set(MessageRecord::getDelTime, DateUtil.now());

        return super.update(qw);
    }


    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取企业详情
     * @param: [id]
     * @return: data.messageRecord.MessageRecord
     **/
    @Override
    public MessageRecord getDetails(String id) {
        return super.getById(id);
    }

}
