package com.jxjy.system.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.jxjy.system.bean.auth.UserContext;
import com.jxjy.system.mapper.IndustryMapper;
import com.jxjy.system.service.IndustryService;
import com.jxjy.system.util.auth.UserUtils;
import com.jxjy.system.xsnowflake.SnowFlakeFactory;
import data.system.Industry;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 *
 * @return: 行业管理
 * @Author: wml
 * @Date 2025/5/9 10:22
 */
@Service
@Transactional
public class IndustryServiceImpl extends ServiceImpl<IndustryMapper, Industry> implements IndustryService {

    @Autowired
    SnowFlakeFactory snowFlakeFactory;


    /**
     * @description: 获取行业列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.industry.Industry>
     **/
    @Override
    public List<Industry> list(int pageNum, int pageSize, Industry industry) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        //创建查询条件
        LambdaQueryWrapper<Industry> qw = new LambdaQueryWrapper<>();
        qw.like(StringUtils.isNotBlank(industry.getName()),Industry::getName,industry.getName())
                .eq(Industry::getDel,false)
                .eq(StringUtils.isNotBlank(industry.getBaseCode()),Industry::getBaseCode, industry.getBaseCode())
                .orderByDesc(Industry::getCreateTime);
        List<Industry> industryList = super.list(qw);
        //返回查询结果
        return industryList;
    }

    /**
     * @description: 获取行业列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.industry.Industry>
     **/
    @Override
    /**
     * 首次调用时自动写入缓存，后续直接从缓存获取
     * value：缓存名称（相当于前缀）
     * key：缓存 key（使用 SpEL 表达式，这里为 ID）
     * expire：过期时间（需配合缓存配置）
     */
    @Cacheable(
            value = "industryList",
            key = "'all'"  // 全量列表用固定 key
            , unless = "#result == null"
    )
    public List<Industry> listAll(Industry industry) {
        //创建查询条件
        LambdaQueryWrapper<Industry> qw = new LambdaQueryWrapper<>();
        qw.eq(Industry::getDel,false)
                .eq(Industry::getParentId, "0")
                .orderByAsc(Industry::getSort);
        List<Industry> industryList = super.list(qw);

        industryList.stream().forEach(industry1 -> {

            //创建查询条件
            LambdaQueryWrapper<Industry> qw1 = new LambdaQueryWrapper<>();
            qw1.eq(Industry::getDel,false)
                    .eq(Industry::getParentId, industry1.getId())
                    .orderByAsc(Industry::getSort);
            List<Industry> industryList1 = super.list(qw1);

            /*industryList1.stream().forEach(industry2 -> {

                //创建查询条件
                LambdaQueryWrapper<Industry> qw2 = new LambdaQueryWrapper<>();
                qw2.eq(Industry::getDel,false)
                        .eq(Industry::getParentId, industry2.getId())
                        .orderByAsc(Industry::getSort);
                List<Industry> industryList2 = super.list(qw2);

                industry2.setChilList(industryList2);

            });*/

            industry1.setChilList(industryList1);

        });

        //返回查询结果
        return industryList;
    }


    // 清空全量列表的缓存（仅清理 key = 'all' 的缓存项）
    @Override
    @CacheEvict(value = "industryList", key = "'all'")
    public void cleanAllIndustryCache() {
        // 方法体可空，仅用于触发缓存清理
        System.out.println("全量列表缓存已清理");
    }


    /**
     * @description: 新增行业
     * @author: wangmenglong
     * @date; 2023/12/11 16:51
     * @param: [industry]
     * @return: boolean
     **/
    @Override
    public boolean create(Industry industry) {

        UserContext userContext = UserUtils.getUserContext();
        String baseCode = userContext.getBaseCode();

        industry.setId(snowFlakeFactory.nextId());
        industry.setBaseCode(baseCode);
        industry.setDel(false);
        industry.setCreateTime(DateUtil.now());

        return super.save(industry);
    }

    /**
     * @description: 编辑行业
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [industry]
     * @return: boolean
     **/
    @Override
    public boolean update(Industry industry) {
        industry.setUpdateTime(DateUtil.now());
        //判断是否传入id
        if(StringUtils.isBlank(industry.getId())){
            return false;
        }
        return super.saveOrUpdate(industry);
    }

    /**
     * @description: 删除行业
     * @author: wangmenglong
     * @date; 2023/12/11 17:06
     * @param: [id]
     * @return: boolean
     **/
    @Override
    public boolean del(String id) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        LambdaUpdateWrapper<Industry> qw = new LambdaUpdateWrapper<>();
        qw.eq(Industry::getId,id)
                .set(Industry::getDel,true)
                .set(Industry::getDelTime, DateUtil.now());

        return super.update(qw);
    }


    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取行业详情
     * @param: [id]
     * @return: data.industry.Industry
     **/
    @Override
    public Industry getDetails(String id) {
        return super.getById(id);
    }

    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取行业详情
     * @param: [id]
     * @return: data.industry.Industry
     **/
    @Override
    public Industry getDetailsByName(String name,String parentType) {

        LambdaQueryWrapper<Industry> qw = new LambdaQueryWrapper<>();
        qw.eq(Industry::getDel,false)
                .eq(Industry::getName, name);

        if(parentType.equals("0")){
            qw.eq(Industry::getParentId, "0");
        }else if(parentType.equals("1")){
            qw.ne(Industry::getParentId, "0");
            qw.isNull(Industry::getCode);
        }

        return super.getOne(qw);
    }

    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取行业详情
     * @param: [id]
     * @return: data.industry.Industry
     **/
    @Override
    public Industry getDetailsByCode(String code,String parentType) {

        LambdaQueryWrapper<Industry> qw = new LambdaQueryWrapper<>();
        qw.eq(Industry::getDel,false)
                .eq(Industry::getCode, code);

        if(parentType.equals("0")){
            qw.eq(Industry::getParentId, "0");
        }else if(parentType.equals("1")){
            qw.ne(Industry::getParentId, "0");
            qw.isNull(Industry::getCode);
        }

        return super.getOne(qw);
    }


    /**
     * @author: wangmenglong
     * @date; 2024/1/2 14:30
     * @description: 判断是否存在
     * @param: [account, id]
     * @return: boolean
     **/
    @Override
    public boolean isHave(String name, String id) {
        LambdaQueryWrapper<Industry> qw = new LambdaQueryWrapper<>();
        qw.eq(Industry::getDel,false)
                .eq(Industry::getName, name);
        //如果传了id就是编辑，编辑的时候要加个条件，不能把名字跟别人重复
        if(StringUtils.isNotBlank(id)){
            qw.ne(Industry::getId,id);
        }
        return super.count(qw)>0;
    }
}
