package com.jxjy.system.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.jxjy.system.bean.auth.UserContext;
import com.jxjy.system.mapper.RecruitMapper;
import com.jxjy.system.service.*;
import com.jxjy.system.util.auth.UserUtils;
import com.jxjy.system.xsnowflake.SnowFlakeFactory;
import data.system.Industry;
import data.system.Message;
import data.system.Org;
import data.system.Recruit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import com.alipay.api.AlipayClient;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.domain.AlipayEbppIndustryJobSyncModel;
import com.alipay.api.request.AlipayEbppIndustryJobSyncRequest;
import com.alipay.api.domain.JobAddress;
import com.alipay.api.response.AlipayEbppIndustryJobSyncResponse;

import javax.annotation.Resource;
import java.util.List;

/**
 *
 * @return: 招聘管理
 * @Author: wml
 * @Date 2025/5/9 10:22
 */
@Service
@Transactional
public class RecruitServiceImpl extends ServiceImpl<RecruitMapper, Recruit> implements RecruitService {

    @Autowired
    SnowFlakeFactory snowFlakeFactory;

    @Resource
    private RecruitMapper recruitMapper;

    @Resource
    private PersonService personService;

    @Resource
    private OrgService orgService;

    @Resource
    private IndustryService industryService;

    @Resource
    private MessageService messageService;


    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> listBySignRecordAdmin(int pageNum, int pageSize,String personIdno,String personName,String name,String personPhoneNumber) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);

        String adminType = "admin";

        UserContext userContext = UserUtils.getUserContext();
        if(!userContext.getType().equals("admin")){
            adminType = userContext.getType();
        }

        List<Recruit> orgList = recruitMapper.listBySignRecord(null,personIdno,personName,name,personPhoneNumber,adminType,userContext.getUserId());

        orgList.stream().forEach(recruit -> {
            recruit.setPerson(personService.getDetails(recruit.getPersonId()));
            recruit.setOrg(orgService.getDetails(recruit.getOrgId()));
        });


        //返回查询结果
        return orgList;
    }

    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> listBySignRecord(int pageNum, int pageSize, String personId) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        List<Recruit> orgList = recruitMapper.listBySignRecord(personId,null,null,null,null,null,null);

        orgList.stream().forEach(returnRecruit -> {
            returnRecruit.setOrg(orgService.getDetails(returnRecruit.getOrgId()));
        });
        //返回查询结果
        return orgList;
    }

    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> listByCollect(int pageNum, int pageSize, String personId) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        List<Recruit> orgList = recruitMapper.listByCollect(personId);

        orgList.stream().forEach(returnRecruit -> {
            returnRecruit.setOrg(orgService.getDetails(returnRecruit.getOrgId()));
        });
        //返回查询结果
        return orgList;
    }

    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> list(int pageNum, int pageSize, Recruit recruit) {

        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        //创建查询条件
        LambdaQueryWrapper<Recruit> qw = new LambdaQueryWrapper<>();
        qw.like(StringUtils.isNotBlank(recruit.getName()),Recruit::getName,recruit.getName())
                .eq(StringUtils.isNotBlank(recruit.getRecruitTypeId()),Recruit::getRecruitTypeId,recruit.getRecruitTypeId())
                .eq(StringUtils.isNotBlank(recruit.getOrgId()),Recruit::getOrgId,recruit.getOrgId())
                .eq(StringUtils.isNotBlank(recruit.getCity()),Recruit::getCity,recruit.getCity())
                .eq(StringUtils.isNotBlank(recruit.getProvince()),Recruit::getProvince,recruit.getProvince())
                .eq(StringUtils.isNotBlank(recruit.getExperienceId()),Recruit::getExperienceId,recruit.getExperienceId())
                .eq(StringUtils.isNotBlank(recruit.getStatus()),Recruit::getStatus,recruit.getStatus())
                .eq(StringUtils.isNotBlank(recruit.getCheckStatus()),Recruit::getCheckStatus,recruit.getCheckStatus())
                .eq(StringUtils.isNotBlank(recruit.getQualificationId()),Recruit::getQualificationId,recruit.getQualificationId())
                .eq(Recruit::getDel,false)
                .eq(StringUtils.isNotBlank(recruit.getBaseCode()),Recruit::getBaseCode, recruit.getBaseCode())
                .eq(StringUtils.isNotBlank(recruit.getUserId()),Recruit::getUserId,recruit.getUserId())
                .ne(StringUtils.isNotBlank(recruit.getType()),Recruit::getType,recruit.getType())
                .like(StringUtils.isNotBlank(recruit.getTowIndustryName()),Recruit::getTowIndustryName, recruit.getTowIndustryName())
                .and(StringUtils.isNotBlank(recruit.getSelTitle()),
                        qw1 -> qw1.like(Recruit::getName,recruit.getSelTitle()).or().like(Recruit::getOrgName,recruit.getSelTitle()))
                .orderByDesc(Recruit::getCreateTime);

        List<Recruit> orgList = super.list(qw);

        orgList.stream().forEach(returnRecruit -> {
            returnRecruit.setOrg(orgService.getDetails(returnRecruit.getOrgId()));
        });

        //返回查询结果
        return orgList;
    }


    /**
     * @description: 新增招聘
     * @author: wangmenglong
     * @date; 2023/12/11 16:51
     * @param: [recruit]
     * @return: boolean
     **/
    @Override
    public boolean create(Recruit recruit) {

        UserContext userContext = UserUtils.getUserContext();
        String baseCode = userContext.getBaseCode();

        recruit.setId(snowFlakeFactory.nextId());
        recruit.setBaseCode(baseCode);
        recruit.setDel(false);
        recruit.setCreateTime(DateUtil.now());
        recruit.setStatus("down");
        recruit.setLaunch("0");
        recruit.setUserId(userContext.getUserId());
        recruit.setUserName(userContext.getUsername());

        if(userContext.getType().equals("admin")){
            recruit.setCheckStatus("1");
            recruit.setType("admin");
        }else if(!userContext.getType().equals("admin")){
            recruit.setType(userContext.getType());
            recruit.setCheckStatus("0");
        }

        try {

            Industry industry = industryService.getDetails(recruit.getIndustryId());

            recruit.setIndustryCode(industry.getCode());


        }catch (Exception e){
            e.printStackTrace();
        }

        return super.save(recruit);
    }

    /**
     * @description: 编辑招聘
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [org]
     * @return: boolean
     **/
    @Override
    public boolean update(Recruit recruit) {
        recruit.setUpdateTime(DateUtil.now());
        //判断是否传入id
        if(StringUtils.isBlank(recruit.getId())){
            return false;
        }
        return super.saveOrUpdate(recruit);
    }

    /**
     * @description: 删除招聘
     * @author: wangmenglong
     * @date; 2023/12/11 17:06
     * @param: [id]
     * @return: boolean
     **/
    @Override
    public boolean del(String id) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        LambdaUpdateWrapper<Recruit> qw = new LambdaUpdateWrapper<>();
        qw.eq(Recruit::getId,id)
                .set(Recruit::getDel,true)
                .set(Recruit::getDelTime, DateUtil.now());

        return super.update(qw);
    }


    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取招聘详情
     * @param: [id]
     * @return: data.recruit.Recruit
     **/
    @Override
    public Recruit getDetails(String id) {

        Recruit recruit = super.getById(id);

        recruit.setOrg(orgService.getDetails(recruit.getOrgId()));

        return super.getById(id);
    }


    /**
     * @description: 编辑招聘
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [org]
     * @return: boolean
     **/
    @Override
    public boolean updateStatus(Recruit recruit) {
        //判断是否传入id
        if(StringUtils.isBlank(recruit.getId())){
            return false;
        }

        recruit = super.getById(recruit.getId());

        recruit.setOrg(orgService.getDetails(recruit.getOrgId()));

        switch (recruit.getStatus()){
            case "up":
                recruit.setStatus("down");
                recruit.setLaunch("1");
                break;
            case "down":
                recruit.setStatus("up");
                break;
            default:
                recruit.setStatus("up");
                return false;
        }

        //发送消息
        if(recruit.getStatus().equals("up")){

            Message message = new Message();
            message.setTitle("新增岗位发布");
            message.setContent("您关注的岗位类型有新的岗位发布，点击前往查看。");
            message.setRecruitId(recruit.getId());
            messageService.create(message);

        }

        return super.saveOrUpdate(recruit);
    }

    /**
     * @description: 编辑招聘
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [org]
     * @return: boolean
     **/
    @Override
    public boolean checkStatus(Recruit recruit) {
        //判断是否传入id
        if(StringUtils.isBlank(recruit.getId())){
            return false;
        }

        Recruit selrecruit = super.getById(recruit.getId());

        selrecruit.setCheckStatus(recruit.getCheckStatus());
        selrecruit.setCheckRemark(recruit.getCheckRemark());

        return super.saveOrUpdate(selrecruit);
    }

}
