import axios from 'axios'
import { ElMessageBox, ElMessage } from 'element-plus'
import { getToken } from '@/utils/auth'
import errorCode from '@/utils/errorCode'
import { tansParams,objectToQueryString } from '@/utils/ruoyi'
import cache from '@/plugins/cache'
import useUserStore from '@/store/modules/user'

// 是否显示重新登录
export let isRelogin = { show: false };

axios.defaults.headers['Content-Type'] = 'application/json;charset=utf-8'
// 创建axios实例
const service = axios.create({
  // axios中请求配置有baseURL选项，表示请求URL公共部分
  baseURL: import.meta.env.VITE_APP_BASE_API,
  // 超时
  timeout: 10 * 1000
})
// post 入参转FormData
const setFormData = (data) => {
  const formData = new FormData()
  const keys = Object.keys(data)
  for(let i = 0;i<keys.length;i++) {
    if (data[keys[i]]!=null&&[keys[i]]!=undefined&&data[keys[i]].constructor ==  Array){
    // 数组参数转FormData
      for(let j = 0; j < data[keys[i]].length; j++) {
        formData.append(keys[i], data[keys[i]][j])
      }
    } else {
      formData.append(keys[i], data[keys[i]])
    }
  }
  return formData
}
// 去除入参中的null值
const removeNullValues =(obj) => {
  for (let key in obj) {
      if (obj[key] === null) {
          delete obj[key];
      } else if (typeof obj[key] === 'object') {
          removeNullValues(obj[key]);
      }
  }
  return obj;
}
// request拦截器
service.interceptors.request.use(config => {
  // if (config.method === 'post' && config.url!=='/djl-account/role/c' && config.url!=='/djl-account/role/u') {
  //   config.data = removeNullValues(config.data)
  //   config.data = setFormData(config.data)
  // }
  // 是否需要设置 token
  const isToken = (config.headers || {}).isToken === false
  // 是否需要防止数据重复提交
  const isRepeatSubmit = (config.headers || {}).repeatSubmit === false
  if (getToken() && !isToken) {
    config.headers['Authorization'] = 'Bearer ' + getToken() // 让每个请求携带自定义token 请根据实际情况自行修改
  }
  // get请求映射params参数
  if (config.method === 'get' && config.params) {
    let url = config.url + '?' + tansParams(config.params);
    url = url.slice(0, -1);
    config.params = {};
    config.url = url;
  }
  if (!isRepeatSubmit && (config.method === 'post' || config.method === 'put')) {
    const requestObj = {
      url: config.url,
      data: typeof config.data === 'object' ? JSON.stringify(config.data) : config.data,
      time: new Date().getTime()
    }
    const requestSize = Object.keys(JSON.stringify(requestObj)).length; // 请求数据大小
    const limitSize = 5 * 1024 * 1024; // 限制存放数据5M
    if (requestSize >= limitSize) {
      console.warn(`[${config.url}]: ` + '请求数据大小超出允许的5M限制，无法进行防重复提交验证。')
      return config;
    }
    const sessionObj = cache.session.getJSON('sessionObj')
    if (sessionObj === undefined || sessionObj === null || sessionObj === '') {
      cache.session.setJSON('sessionObj', requestObj)
    } else {
      const s_url = sessionObj.url;                // 请求地址
      const s_data = sessionObj.data;              // 请求数据
      const s_time = sessionObj.time;              // 请求时间
      const interval = 1000;                       // 间隔时间(ms)，小于此时间视为重复提交
      if (s_data === requestObj.data && requestObj.time - s_time < interval && s_url === requestObj.url) {
        const message = '数据正在处理，请勿重复提交';
        console.warn(`[${s_url}]: ` + message)
        return Promise.reject(new Error(message))
      } else {
        cache.session.setJSON('sessionObj', requestObj)
      }
    }
  }
  return config
}, error => {
    console.log(error)
    Promise.reject(error)
})

// 响应拦截器
service.interceptors.response.use(res => {
    // 未设置状态码则默认成功状态
    if (res.data.isSuccess==undefined||res.data.isSuccess) {
      let errorMsg = '';
      if(res.data.code===101){
        errorMsg = '您的账号已被禁用，2 秒后将调回登录页';
      }else if(res.data.code===102){
        errorMsg = '账号不存在，2 秒后将调回登录页';
      }else if(res.data.code===103){
        errorMsg = '角色错误，2 秒后将调回登录页';
      }else if(res.data.code===401){
        errorMsg = '未登录，2 秒后将调回登录页';
      }else {
        return Promise.resolve(res.data)
      }
      ElMessageBox.alert(errorMsg, '系统提示', {type: 'warning'});
      setTimeout(() => {
        useUserStore().logOut().then(() => {
          location.href = '#/login';
        });
      }, 2000);

    } else {
      const code = res.data.code || '';
      const msg = errorCode[code] || res.data.data || errorCode['default']
      if (code === 401) {
        if (!isRelogin.show) {
          isRelogin.show = true;
          ElMessageBox.confirm('登录状态已过期，您可以继续留在该页面，或者重新登录', '系统提示', { confirmButtonText: '重新登录', cancelButtonText: '取消', type: 'warning' }).then(() => {
            isRelogin.show = false;
            useUserStore().logOut().then(() => {
              location.href = '#/login';
            })
          }).catch(() => {
            isRelogin.show = false;
          });
        }
        return Promise.reject('无效的会话，或者会话已过期，请重新登录。')
      }
      ElMessage({ message: msg, type: 'error' })
      return Promise.reject(new Error(msg)) 
    }
  },
  error => {
    console.log('err' + error)
    let { message } = error;
    if (message == "Network Error") {
      message = "后端接口连接异常";
    } else if (message.includes("timeout")) {
      message = "系统接口请求超时";
    } else if (message.includes("Request failed with status code")) {
      message = "系统接口" + message.substr(message.length - 3) + "异常";
    }
    ElMessage({ message: message, type: 'error', duration: 5 * 1000 })
    return Promise.reject(error)
  }
)

// 通用下载方法
export function download(url,params,filename, type = "POST") {
  let xhr = new XMLHttpRequest();
  // params = {}
  let param = objectToQueryString(params)
  url = import.meta.env.VITE_APP_BASE_API + url
  if (param) {
    url = url + '?' + param
  }
  xhr.open(type, url);
  xhr.responseType = "blob";
  xhr.setRequestHeader("Authorization", 'Bearer ' + getToken())
  xhr.onload = () => {
    let ctx = xhr.response;
    let blob = new Blob([ctx]);
    if ("msSaveOrOpenBlob" in navigator) {//兼容IE
      window.navigator.msSaveOrOpenBlob(blob, filename);
    } else {
      let aLink = document.createElement("a");
      aLink.download = filename;
      aLink.style.display = "none";
      aLink.href = URL.createObjectURL(blob);
      document.body.appendChild(aLink);
      aLink.click();
      document.body.removeChild(aLink);
    }
  };
  xhr.send();
}

export default service
