package com.jfb.recruit.util.Excelutils;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * 适配EasyExcel 2.2.10 + POI 3.17的Excel追加工具类（彻底解决空文件+覆盖问题）
 */
public class OrgExcelAppendUtil {

    // 获取桌面Excel路径
    public static String getDesktopExcelPath(String excelFileName) {
        String desktopPath = System.getProperty("user.home") + File.separator + "Desktop";
        return desktopPath + File.separator + excelFileName;
    }

    // 读取Excel中所有已有数据（2.2.10兼容写法）
    private static List<OrgExcel> readAllExcelData(String excelPath) {
        File excelFile = new File(excelPath);
        // 空文件/不存在直接返回空列表
        if (!excelFile.exists() || excelFile.length() == 0) {
            return new ArrayList<>();
        }

        // 自定义监听器读取所有数据
        class DataReadListener extends AnalysisEventListener<OrgExcel> {
            private List<OrgExcel> dataList = new ArrayList<>();

            @Override
            public void invoke(OrgExcel data, AnalysisContext context) {
                dataList.add(data);
            }

            @Override
            public void doAfterAllAnalysed(AnalysisContext context) {}

            public List<OrgExcel> getDataList() {
                return dataList;
            }
        }

        DataReadListener listener = new DataReadListener();
        try {
            EasyExcel.read(excelPath, OrgExcel.class, listener)
                    .sheet("上传记录")
                    .doRead();
        } catch (Exception e) {
            System.out.println("读取Excel数据失败，将重新创建文件：" + e.getMessage());
            return new ArrayList<>();
        }
        return listener.getDataList();
    }

    // 核心：追加数据（2.2.10兼容，零覆盖+空文件修复）
    public static void appendDataToExcel(String excelFileName, List<OrgExcel> newData) {
        String excelPath = getDesktopExcelPath(excelFileName);
        File excelFile = new File(excelPath);

        // ========== 第一步：最前置的空文件检测（关键！拦截所有空文件场景） ==========
        if (excelFile.exists()) {
            // 1. 空文件直接删除，避免POI解析报错
            if (excelFile.length() == 0) {
                excelFile.delete();
                System.out.println("检测到空Excel文件，已删除：" + excelPath);
            }
            // 2. 非空文件但读取失败（损坏），也删除
            else {
                try {
                    // 尝试读取文件头，验证文件有效性
                    EasyExcel.read(excelPath).sheet("上传记录").doRead();
                } catch (Exception e) {
                    excelFile.delete();
                    System.out.println("Excel文件损坏，已删除：" + e.getMessage());
                }
            }
        }

        try {
            // ========== 第二步：读取原有数据 + 合并新数据 ==========
            List<OrgExcel> allData = new ArrayList<>();
            // 文件存在且有效：读取原有数据
            if (excelFile.exists() && excelFile.length() > 0) {
                allData = readAllExcelData(excelPath);
                System.out.println("读取到原有Excel数据行数：" + allData.size());
            }
            // 合并新数据（核心：追加=原有数据+新数据）
            allData.addAll(newData);

            // ========== 第三步：重新写入（覆盖整个文件，但数据是合并后的，等价于追加） ==========
            EasyExcel.write(excelPath, OrgExcel.class)
                    .sheet("上传记录")
                    .doWrite(allData);

            // ========== 第四步：验证写入结果，避免生成空文件 ==========
            if (!excelFile.exists() || excelFile.length() == 0) {
                throw new RuntimeException("写入Excel失败：生成了空文件（桌面权限不足/文件被占用）");
            }

            System.out.println("数据追加成功！总数据行数：" + allData.size() + "，文件路径：" + excelPath);

        } catch (Exception e) {
            // 异常时清理空文件
            if (excelFile.exists() && excelFile.length() == 0) {
                excelFile.delete();
            }
            throw new RuntimeException("追加Excel数据失败：" + e.getMessage(), e);
        }
    }
}