package com.jfb.recruit.service.impl;

import base.result.BaseResult;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.jfb.recruit.bean.auth.UserContext;
import com.jfb.recruit.mapper.UserMapper;
import com.jfb.recruit.service.UserService;
import com.jfb.recruit.util.auth.UserUtils;
import com.jfb.recruit.xsnowflake.SnowFlakeFactory;
import com.github.pagehelper.PageHelper;
import data.user.User;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.DigestUtils;

import java.util.List;

/**
 *
 * @return: 用户管理
 * @Author: wml
 * @Date 2025/5/9 10:22
 */
@Service
@Transactional
public class UserServiceImpl extends ServiceImpl<UserMapper, User> implements UserService {

    @Autowired
    SnowFlakeFactory snowFlakeFactory;

    /**
    * @author: wangmenglong
    * @date; 2025/5/9 10:22
    * @description: 获取用户详情,登陆用
    * @param: [id]
    * @return: data.user.User
    **/
    @Override
    public User getDetailsByLoginAccount(String account) {

        //创建查询条件
        LambdaQueryWrapper<User> qw = new LambdaQueryWrapper<>();
        qw.eq(User::getAccount,account)
                .eq(User::getDel,false);

        return super.getOne(qw);
    }

    /**
     * @description: 获取用户列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.user.User>
     **/
    @Override
    public List<User> list(int pageNum, int pageSize, User user) {
        UserContext userContext = UserUtils.getUserContext();
        String baseCode = userContext.getBaseCode();
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        //创建查询条件
        LambdaQueryWrapper<User> qw = new LambdaQueryWrapper<>();
        qw.like(StringUtils.isNotBlank(user.getName()),User::getName,user.getName())
                .like(StringUtils.isNotBlank(user.getAccount()),User::getAccount,user.getAccount())
                .eq(StringUtils.isNotBlank(user.getState()), User::getState,user.getState())
                .eq(User::getDel,false)
                .eq(User::getBaseCode, baseCode)
                .orderByDesc(User::getCreateTime);
        List<User> orgList = super.list(qw);
        //返回查询结果
        return orgList;
    }


    /**
     * @description: 新增用户
     * @author: wangmenglong
     * @date; 2023/12/11 16:51
     * @param: [user]
     * @return: boolean
     **/
    @Override
    public boolean create(User user) {
        //校验用户名是否存在
        User u = super.getOne(new LambdaQueryWrapper<User>().eq(User::getDel, false )
                .eq(User::getState, "normal")
                .eq(User::getAccount, user.getAccount()));
        if(ObjectUtil.isNotNull(u)){
            return false;
        }

        UserContext userContext = UserUtils.getUserContext();
        String baseCode = userContext.getBaseCode();

        user.setId(snowFlakeFactory.nextId());
        user.setBaseCode(baseCode);
        user.setUserId(userContext.getUserId());
        user.setUserName(userContext.getUsername());
        user.setPassword(DigestUtils.md5DigestAsHex("123456".getBytes()));
        user.setState("normal");
        user.setDel(false);
        user.setCreateTime(DateUtil.now());

        return super.save(user);
    }

    /**
     * @description: 编辑用户
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [org]
     * @return: boolean
     **/
    @Override
    public boolean update(User user) {
        user.setUpdateTime(DateUtil.now());
        //判断是否传入id
        if(StringUtils.isBlank(user.getId())){
            return false;
        }
        return super.saveOrUpdate(user);
    }

    /**
     * @description: 重置密码
     * @author: wangmenglong
     * @date; 2023/12/11 17:00
     * @param: [id]
     * @return: boolean
     **/
    @Override
    public boolean reset(String id) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        User user = super.getById(id);
        user.setPassword(DigestUtils.md5DigestAsHex("123456".getBytes()));
        user.setUpdateTime(DateUtil.now());
        return super.saveOrUpdate(user);
    }

    /**
     * @description: 修改状态
     * @author: wangmenglong
     * @date; 2023/12/11 17:04
     * @param: [id, state]
     * @return: boolean
     **/
    @Override
    public boolean uState(String id, String state) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        User user = super.getById(id);
        user.setState(state);
        user.setUpdateTime(DateUtil.now());
        return super.saveOrUpdate(user);
    }

    /**
     * @description: 删除用户
     * @author: wangmenglong
     * @date; 2023/12/11 17:06
     * @param: [id]
     * @return: boolean
     **/
    @Override
    public boolean del(String id) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        LambdaUpdateWrapper<User> qw = new LambdaUpdateWrapper<>();
        qw.eq(User::getId,id)
                .set(User::getDel,true)
                .set(User::getDelTime, DateUtil.now());

        User user = super.getById(id);

        return super.update(qw);
    }

    /**
     * @description: 修改密码
     * @author: wangmenglong
     * @date; 2023/12/12 11:05
     * @param: [oldPassword, password, id]
     * @return: boolean
     **/
    @Override
    public boolean uPassword(String oldPassword, String password, String id) {
        User user = super.getById(id);
        oldPassword = DigestUtils.md5DigestAsHex(oldPassword.getBytes());
        if(!oldPassword.equals(user.getPassword())){
            return false;
        }
        password = DigestUtils.md5DigestAsHex(password.getBytes());
        user.setPassword(password);
        user.setUpdateTime(DateUtil.now());

        return super.saveOrUpdate(user);
    }

    /**
     * @author: wangmenglong
     * @date; 2024/3/21 15:39
     * @description: 修改当前用户的密码
     * @param: [oldPassword, password]
     * @return: base.result.BaseResult
     **/
    @Override
    public BaseResult updateUserPassword(String oldPassword, String password) {
        UserContext userContext = UserUtils.getUserContext();
        String id = userContext.getUserId();
        User user = super.getById(id);
        oldPassword = DigestUtils.md5DigestAsHex(oldPassword.getBytes());
        if(!oldPassword.equals(user.getPassword())){
            return BaseResult.error("error_035");
        }
        LambdaUpdateWrapper<User> qw = new LambdaUpdateWrapper<>();
        qw.eq(User::getId,id)
                .set(User::getPassword,DigestUtils.md5DigestAsHex(password.getBytes()))
                .set(User::getUpdateTime, DateUtil.now());
        return super.update(qw) ? BaseResult.success("normal_002") : BaseResult.error("error_002");
    }

    /**
     * @author: wangmenglong
     * @date; 2024/1/2 14:30
     * @description: 判断用户名是否存在
     * @param: [account, id]
     * @return: boolean
     **/
    @Override
    public boolean isHave(String account, String id) {
        LambdaQueryWrapper<User> qw = new LambdaQueryWrapper<>();
        qw.eq(User::getDel,false)
                .eq(User::getState, "normal")
                .eq(User::getAccount,account);
        //如果传了id就是编辑，编辑的时候要加个条件，不能把名字跟别人重复
        if(StringUtils.isNotBlank(id)){
            qw.ne(User::getId,id);
        }
        return super.count(qw)>0;
    }



    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取用户详情
     * @param: [id]
     * @return: data.user.User
     **/
    @Override
    public User getDetails(String id) {
        return super.getById(id);
    }

}
