package com.jfb.recruit.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.jfb.recruit.bean.auth.UserContext;
import com.jfb.recruit.mapper.ObjectiveMapper;
import com.jfb.recruit.service.ObjectiveService;
import com.jfb.recruit.service.PersonService;
import com.jfb.recruit.util.auth.UserUtils;
import com.jfb.recruit.xsnowflake.SnowFlakeFactory;
import data.recruit.Objective;
import data.recruit.Person;
import data.recruit.WorkExperience;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

/**
 *
 * @return: 求职意向管理
 * @Author: wml
 * @Date 2025/5/9 10:22
 */
@Service
@Transactional
public class ObjectiveServiceImpl extends ServiceImpl<ObjectiveMapper, Objective> implements ObjectiveService {

    @Autowired
    SnowFlakeFactory snowFlakeFactory;

    @Resource
    private PersonService personService;

    /**
     * @description: 获取求职意向列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.objective.Objective>
     **/
    @Override
    public List<Objective> list(int pageNum, int pageSize, Objective objective) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        //创建查询条件
        LambdaQueryWrapper<Objective> qw = new LambdaQueryWrapper<>();
        qw.eq(Objective::getDel,false)
                .eq(StringUtils.isNotBlank(objective.getBaseCode()),Objective::getBaseCode, objective.getBaseCode())
                .eq(StringUtils.isNotBlank(objective.getPersonId()),Objective::getPersonId, objective.getPersonId())
                .eq(StringUtils.isNotBlank(objective.getIndustryId()), Objective::getIndustryId, objective.getIndustryId())
                .like(StringUtils.isNotBlank(objective.getIndustryName()),Objective::getIndustryName, objective.getIndustryName())
                .orderByDesc(Objective::getCreateTime);
        List<Objective> objectiveList = super.list(qw);

        objectiveList.stream().forEach(objective1 -> {
            objective1.setPerson(personService.getDetails(objective1.getPersonId()));
        });

        //返回查询结果
        return objectiveList;
    }


    /**
     * @description: 新增求职意向
     * @author: wangmenglong
     * @date; 2023/12/11 16:51
     * @param: [objective]
     * @return: boolean
     **/
    @Override
    public boolean create(Objective objective) {

        UserContext userContext = UserUtils.getUserContext();
        String baseCode = userContext.getBaseCode();

        objective.setId(snowFlakeFactory.nextId());
        objective.setBaseCode(baseCode);
        objective.setDel(false);
        objective.setCreateTime(DateUtil.now());

        return super.save(objective);
    }

    /**
     * @description: 编辑求职意向
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [objective]
     * @return: boolean
     **/
    @Override
    public boolean update(Objective objective) {
        objective.setUpdateTime(DateUtil.now());
        //判断是否传入id
        if(StringUtils.isBlank(objective.getId())){
            return false;
        }
        return super.saveOrUpdate(objective);
    }

    /**
     * @description: 删除求职意向
     * @author: wangmenglong
     * @date; 2023/12/11 17:06
     * @param: [id]
     * @return: boolean
     **/
    @Override
    public boolean del(String id) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        LambdaUpdateWrapper<Objective> qw = new LambdaUpdateWrapper<>();
        qw.eq(Objective::getId,id)
                .set(Objective::getDel,true)
                .set(Objective::getDelTime, DateUtil.now());

        return super.update(qw);
    }


    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取求职意向详情
     * @param: [id]
     * @return: data.objective.Objective
     **/
    @Override
    public Objective getDetails(String id) {
        return super.getById(id);
    }

    /**
     * @author: wangmenglong
     * @date; 2024/1/2 14:30
     * @description: 判断是否存在
     * @param: [account, id]
     * @return: boolean
     **/
    @Override
    public boolean isHave(String userId, String industryId,String id) {
        LambdaQueryWrapper<Objective> qw = new LambdaQueryWrapper<>();
        qw.eq(Objective::getDel,false)
                .eq(Objective::getIndustryId, industryId)
                .eq(Objective::getUserId, userId);
        //如果传了id就是编辑，编辑的时候要加个条件，不能把名字跟别人重复
        if(StringUtils.isNotBlank(id)){
            qw.ne(Objective::getId,id);
        }
        return super.count(qw)>0;
    }

    /**
     * @description: 新增求职意向
     * @author: wangmenglong
     * @date; 2023/12/11 16:51
     * @param: [workExperience]
     * @return: boolean
     **/
    @Override
    public boolean createAli(List<Objective> objectives) {

        LambdaUpdateWrapper<Objective> qw = new LambdaUpdateWrapper<>();
        qw.eq(Objective::getDel,false)
                .eq(Objective::getPersonId, objectives.get(0).getPersonId())
                .set(Objective::getDel,true);
        super.update(qw);

        objectives.stream().forEach(objective -> {
            objective.setId(snowFlakeFactory.nextId());
            objective.setDel(false);
            objective.setCreateTime(DateUtil.now());
        });

        return super.saveBatch(objectives);
    }

}
