package com.jfb.recruit.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.jfb.recruit.bean.auth.UserContext;
import com.jfb.recruit.controller.api.ApiAliController;
import com.jfb.recruit.mapper.RecruitMapper;
import com.jfb.recruit.service.IndustryService;
import com.jfb.recruit.service.OrgService;
import com.jfb.recruit.service.PersonService;
import com.jfb.recruit.service.RecruitService;
import com.jfb.recruit.util.auth.UserUtils;
import com.jfb.recruit.xsnowflake.SnowFlakeFactory;
import data.recruit.Industry;
import data.recruit.Org;
import data.recruit.Recruit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.AlipayConfig;
import com.alipay.api.domain.AlipayEbppIndustryJobSyncModel;
import com.alipay.api.request.AlipayEbppIndustryJobSyncRequest;
import com.alipay.api.domain.JobAddress;
import com.alipay.api.response.AlipayEbppIndustryJobSyncResponse;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @return: 招聘管理
 * @Author: wml
 * @Date 2025/5/9 10:22
 */
@Service
@Transactional
public class RecruitServiceImpl extends ServiceImpl<RecruitMapper, Recruit> implements RecruitService {

    @Autowired
    SnowFlakeFactory snowFlakeFactory;

    @Resource
    private RecruitMapper recruitMapper;

    @Resource
    private PersonService personService;

    @Resource
    private OrgService orgService;

    @Resource
    private IndustryService industryService;


    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> listBySignRecordAdmin(int pageNum, int pageSize,String personIdno,String personName,String name) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        List<Recruit> orgList = recruitMapper.listBySignRecord(null,personIdno,personName,name);

        orgList.stream().forEach(recruit -> {
            recruit.setPerson(personService.getDetails(recruit.getPersonId()));
            recruit.setOrg(orgService.getDetails(recruit.getOrgId()));
        });


        //返回查询结果
        return orgList;
    }

    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> listBySignRecord(int pageNum, int pageSize, String personId) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        List<Recruit> orgList = recruitMapper.listBySignRecord(personId,null,null,null);

        orgList.stream().forEach(returnRecruit -> {
            returnRecruit.setOrg(orgService.getDetails(returnRecruit.getOrgId()));
        });
        //返回查询结果
        return orgList;
    }

    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> listByCollect(int pageNum, int pageSize, String personId) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        List<Recruit> orgList = recruitMapper.listByCollect(personId);

        orgList.stream().forEach(returnRecruit -> {
            returnRecruit.setOrg(orgService.getDetails(returnRecruit.getOrgId()));
        });
        //返回查询结果
        return orgList;
    }

    /**
     * @description: 获取招聘列表
     * @author: wangmenglong
     * @date; 2023/12/11 16:46
     * @param: [pageNum, pageSize]
     * @return: java.util.List<data.recruit.Recruit>
     **/
    @Override
    public List<Recruit> list(int pageNum, int pageSize, Recruit recruit) {
        //启动分页工具
        PageHelper.startPage(pageNum, pageSize);
        //创建查询条件
        LambdaQueryWrapper<Recruit> qw = new LambdaQueryWrapper<>();
        qw.like(StringUtils.isNotBlank(recruit.getName()),Recruit::getName,recruit.getName())
                .eq(StringUtils.isNotBlank(recruit.getRecruitTypeId()),Recruit::getRecruitTypeId,recruit.getRecruitTypeId())
                .eq(StringUtils.isNotBlank(recruit.getOrgId()),Recruit::getOrgId,recruit.getOrgId())
                .eq(StringUtils.isNotBlank(recruit.getCity()),Recruit::getCity,recruit.getCity())
                .eq(StringUtils.isNotBlank(recruit.getProvince()),Recruit::getProvince,recruit.getProvince())
                .eq(StringUtils.isNotBlank(recruit.getExperienceId()),Recruit::getExperienceId,recruit.getExperienceId())
                .eq(StringUtils.isNotBlank(recruit.getStatus()),Recruit::getStatus,recruit.getStatus())
                .eq(StringUtils.isNotBlank(recruit.getQualificationId()),Recruit::getQualificationId,recruit.getQualificationId())
                .eq(Recruit::getDel,false)
                .eq(StringUtils.isNotBlank(recruit.getBaseCode()),Recruit::getBaseCode, recruit.getBaseCode())
                .like(StringUtils.isNotBlank(recruit.getIndustryName()),Recruit::getIndustryName, recruit.getIndustryName())
                .and(StringUtils.isNotBlank(recruit.getSelTitle()),
                        qw1 -> qw1.like(Recruit::getName,recruit.getSelTitle()).or().like(Recruit::getOrgName,recruit.getSelTitle()))
                .orderByDesc(Recruit::getCreateTime);
        List<Recruit> orgList = super.list(qw);

        orgList.stream().forEach(returnRecruit -> {
            returnRecruit.setOrg(orgService.getDetails(returnRecruit.getOrgId()));
        });

        //返回查询结果
        return orgList;
    }


    /**
     * @description: 新增招聘
     * @author: wangmenglong
     * @date; 2023/12/11 16:51
     * @param: [recruit]
     * @return: boolean
     **/
    @Override
    public boolean create(Recruit recruit) {

        UserContext userContext = UserUtils.getUserContext();
        String baseCode = userContext.getBaseCode();

        recruit.setId(snowFlakeFactory.nextId());
        recruit.setBaseCode(baseCode);
        recruit.setDel(false);
        recruit.setCreateTime(DateUtil.now());
        recruit.setStatus("down");
        recruit.setLaunch("0");

        try {

            Industry industry = industryService.getDetails(recruit.getIndustryId());

            recruit.setIndustryCode(industry.getCode());

            Org org = orgService.getDetails(recruit.getOrgId());
            recruit.setOrg(org);

        }catch (Exception e){
            e.printStackTrace();
        }

        return super.save(recruit);
    }

    /**
     * @description: 编辑招聘
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [org]
     * @return: boolean
     **/
    @Override
    public boolean update(Recruit recruit) {
        recruit.setUpdateTime(DateUtil.now());
        //判断是否传入id
        if(StringUtils.isBlank(recruit.getId())){
            return false;
        }
        return super.saveOrUpdate(recruit);
    }

    /**
     * @description: 删除招聘
     * @author: wangmenglong
     * @date; 2023/12/11 17:06
     * @param: [id]
     * @return: boolean
     **/
    @Override
    public boolean del(String id) {
        if(StringUtils.isBlank(id)){
            return false;
        }
        LambdaUpdateWrapper<Recruit> qw = new LambdaUpdateWrapper<>();
        qw.eq(Recruit::getId,id)
                .set(Recruit::getDel,true)
                .set(Recruit::getDelTime, DateUtil.now());

        return super.update(qw);
    }


    /**
     * @author: wangmenglong
     * @date; 2024/3/24 17:19
     * @description: 获取招聘详情
     * @param: [id]
     * @return: data.recruit.Recruit
     **/
    @Override
    public Recruit getDetails(String id) {

        Recruit recruit = super.getById(id);

        recruit.setOrg(orgService.getDetails(recruit.getOrgId()));

        return super.getById(id);
    }


    @Override
    public void sendAli(Recruit recruit) throws Exception {
        // 初始化SDK
        AlipayClient alipayClient = new DefaultAlipayClient(ApiAliController.getAlipayConfig());

        // 构造请求参数以调用接口
        AlipayEbppIndustryJobSyncRequest request = new AlipayEbppIndustryJobSyncRequest();
        AlipayEbppIndustryJobSyncModel model = new AlipayEbppIndustryJobSyncModel();

        // 设置商户侧的岗位唯一标识
        model.setOutJobId(recruit.getId());

        // 设置岗位名称
        model.setJobName(recruit.getName());

        // 设置岗位类型
        //model.setJobType("ruanjianceshi");
        model.setJobType(recruit.getIndustryCode());

        // 设置是否在线办公
        model.setWorkOnline(false);

        // 设置岗位详细地址
        JobAddress address = new JobAddress();
        address.setDetail(recruit.getOrg().getAddress());
        address.setRegionCode(recruit.getOrg().getRegionCode());
        model.setAddress(address);

        // 设置岗位的小程序详情页URL
        model.setJobDetailUrl("alipays://platformapi/startapp?appId=2021005184646262&page=pages/home/index&query=");


        // 设置雇主类型
        model.setEmployerType("COMPANY");

        // 设置雇主名称
        model.setEmployerName(recruit.getOrgName());

        // 设置岗位状态
        model.setHireStatus("HIRING");

        // 设置岗位招聘信息生效时间
        model.setStartDate(recruit.getStartDate());

        // 设置岗位招聘信息失效时间
        model.setExpiredDate(recruit.getExpiredDate());

        // 设置年龄范围
        model.setAge("18,60");

        // 设置性别
        model.setGender("ALL");

        // 设置薪资范围
        model.setSalary(recruit.getSalaryMin() + ","+recruit.getSalaryMax());

        // 设置薪资单位
        model.setPayPeriod("月结");

        // 设置招聘人数
        model.setCount((long)recruit.getTimes());
        //model.setCount(1L);

        request.setBizModel(model);
        // 第三方代调用模式下请设置app_auth_token
        // request.putOtherTextParam("app_auth_token", "<-- 请填写应用授权令牌 -->");

        AlipayEbppIndustryJobSyncResponse response = alipayClient.execute(request);
        System.out.println(response.getBody());

        if (response.isSuccess()) {
            System.out.println("调用成功");
        } else {
            System.out.println("调用失败");
            // sdk版本是"4.38.0.ALL"及以上,可以参考下面的示例获取诊断链接
            // String diagnosisUrl = DiagnosisUtils.getDiagnosisUrl(response);
            // System.out.println(diagnosisUrl);
        }
    }

    /**
     * @description: 编辑招聘
     * @author: wangmenglong
     * @date; 2023/12/11 16:55
     * @param: [org]
     * @return: boolean
     **/
    @Override
    public boolean updateStatus(Recruit recruit) {
        //判断是否传入id
        if(StringUtils.isBlank(recruit.getId())){
            return false;
        }

        recruit = super.getById(recruit.getId());
        switch (recruit.getStatus()){
            case "up":
                //todo 发送阿里
                //sendAli(recruit);
                recruit.setStatus("down");
                recruit.setLaunch("1");
                break;
            case "down":
                recruit.setStatus("up");
                break;
            default:
                //todo 发送阿里
                //sendAli(recruit);
                recruit.setStatus("up");
                return false;
        }

        return super.saveOrUpdate(recruit);
    }

}
