package util;


import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @Author wangmenglong
 * @Description 公共工具类
 * @Date 2022/3/8 10:04
 * @Version 1.0
 */
public class CommonUtils {

    /**
    * @Author wangmenglong
    * @Description 去除字符串中的空格、回车、换行符、制表符
    * @Date 2022/3/8 10:22
    * @Param [str]
    * @return java.lang.String
    **/
    public static String replaceBlank(String str) {
        String dest = "";
        if (!StringUtils.isEmpty(str)) {
            Pattern p = Pattern.compile("\\s*|\t|\r|\n");
            Matcher m = p.matcher(str);
            dest = m.replaceAll("");
        }
        return dest;
    }

    /**
    * @author: wangmenglong
    * @date; 2024/1/9 15:26
    * @description: 计算当前时间距离明天凌晨0时0分0秒的秒数
    * @param: [currentDate]
    * @return: java.lang.Integer
    **/
    public static Integer getRemainSecondsOneDay(Date currentDate) {
        //使用plusDays加传入的时间加1天，将时分秒设置成0
        LocalDateTime midnight = LocalDateTime.ofInstant(currentDate.toInstant(),
                        ZoneId.systemDefault()).plusDays(1).withHour(0).withMinute(0)
                .withSecond(0).withNano(0);
        LocalDateTime currentDateTime = LocalDateTime.ofInstant(currentDate.toInstant(),
                ZoneId.systemDefault());
        //使用ChronoUnit.SECONDS.between方法，传入两个LocalDateTime对象即可得到相差的秒数
        long seconds = ChronoUnit.SECONDS.between(currentDateTime, midnight);
        return (int) seconds;
    }

    /**
    * @author: wangmenglong
    * @date; 2024/3/8 14:42
    * @description: 获取两个日期之间所有的日期yyyy-MM-dd
    * @param: [startDate, endDate]
    * @return: java.util.List<java.lang.String>
    **/
    public static List<String> getDateRange(String startDate, String endDate) {
        List<String> dateRange = new ArrayList<>();
        LocalDate startLocalDate = LocalDate.parse(startDate);
        LocalDate endLocalDate = LocalDate.parse(endDate);
        long daysBetween = ChronoUnit.DAYS.between(startLocalDate, endLocalDate);

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        for (int i = 0; i <= daysBetween; i++) {
            LocalDate currentDate = startLocalDate.plusDays(i);
            String localDateNowStr = currentDate.format(formatter);
            dateRange.add(localDateNowStr);
        }
        return dateRange;
    }

    /**
     * @author: wangmenglong
     * @date; 2024/3/8 14:42
     * @description: 获取两个日期之间所有的月份yyyy-MM
     * @param: [startDate, endDate]
     * @return: java.util.List<java.lang.String>
     **/
    public static List<String> getMonthDateRange(String startDate, String endDate) {
        List<String> dateRange = new ArrayList<>();
        LocalDate startLocalDate = LocalDate.parse(startDate);
        LocalDate endLocalDate = LocalDate.parse(endDate);
        long monthsBetween = ChronoUnit.MONTHS.between(startLocalDate, endLocalDate);

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM");
        for (int i = 0; i <= monthsBetween; i++) {
            LocalDate date = startLocalDate.plusMonths(i);
            String localDateNowStr = date.format(formatter);
            dateRange.add(localDateNowStr);
        }
        return dateRange;
    }

    /**
    * @author: wangmenglong
    * @date; 2024/5/15 18:13
    * @description: 比较金额是否在区间范围内
    * @param: [value, lower, upper]
    * @return: boolean
    **/
    public static boolean isInRange(BigDecimal value, BigDecimal lower, BigDecimal upper) {
        int lowerCompare = value.compareTo(lower);
        int upperCompare = value.compareTo(upper);
        return lowerCompare >= 0 && upperCompare <= 0;
    }


}
