package util.excel;

import com.alibaba.fastjson.JSONObject;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.util.CellReference;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 读取excel
 */
public interface ReadExcel  {

    /**
     * 返回结果成功或失败
     */
    public JSONObject execute();

    /**
     * 获取excel列名
     */
    default String getColumnName(int columnIndex) {
        return CellReference.convertNumToColString(columnIndex);
    }

    /**
     * 获取excel单元格信息
     * CellTypeEnum 类型值
     * NUMERIC 数值型0
     * STRING  字符串型1
     * FORMULA  公式型 2
     * BLANK   空值 3
     * BOOLEAN  布尔型4
     * ERROR   错误5
     */
    default String[] getCellForm(Cell cell) {
        if (cell == null) {
            return null;
        }

        String[] form = new String[4];
        String columnName = "";
        String type = "";
        String val = "";
        String formula = "";
        switch (cell.getCellTypeEnum()) {
            case NUMERIC:
                columnName = getColumnName(cell.getColumnIndex());
                type = "NUMERIC";
                if (HSSFDateUtil.isCellDateFormatted(cell)) {
                    Date d = cell.getDateCellValue();
                    DateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                    val = format.format(d);
                } else {
                    val = String.valueOf(cell.getNumericCellValue());
                }
                break;
            case STRING:
                columnName = getColumnName(cell.getColumnIndex());
                type = "STRING";
                val = cell.getStringCellValue();
                break;
            case FORMULA:
                columnName = getColumnName(cell.getColumnIndex());
                type = "FORMULA";
                val = String.valueOf(cell.getNumericCellValue());
                formula = cell.getCellFormula();
                break;
            case BLANK:
                columnName = getColumnName(cell.getColumnIndex());
                type = "BLANK";
                val = "";
                break;
            case BOOLEAN:
                columnName = getColumnName(cell.getColumnIndex());
                type = "BOOLEAN";
                val = cell.getStringCellValue();
                break;
            case ERROR:
                columnName = getColumnName(cell.getColumnIndex());
                type = "ERROR";
                val = "";
                break;
            default:
                break;
        }

        form[0] = columnName;
        form[1] = type;
        form[2] = val;
        form[3] = formula;
        return form;
    }

    /**
     * 获取excel单元格的值
     * CellTypeEnum 类型值
     * NUMERIC 数值型0
     * STRING  字符串型1
     * FORMULA  公式型 2
     * BLANK   空值 3
     * BOOLEAN  布尔型4
     * ERROR   错误5
     */
    default String getCellVal(Cell cell, FormulaEvaluator formulaEvaluator) {
        if (cell == null) {
            return null;
        }
        String val = "";
        switch (cell.getCellTypeEnum()) {
            case NUMERIC:
                val = String.valueOf(cell.getNumericCellValue());
                break;
            case STRING:
                val = cell.getStringCellValue();
                break;
            case FORMULA:
                val = String.valueOf(formulaEvaluator.evaluate(cell).getNumberValue());
                break;
            case BLANK:
                val = "";
                break;
            case BOOLEAN:
                val = cell.getStringCellValue();
                break;
            case ERROR:
                val = "";
                break;
            default:
                break;
        }
        return val;
    }

    default String getCellVal(Cell cell) {
        if (cell == null) {
            return null;
        }
        String val = "";
        switch (cell.getCellTypeEnum()) {
            case NUMERIC:
                val = String.valueOf(cell.getNumericCellValue());
                break;
            case STRING:
                val = cell.getStringCellValue();
                break;
            case FORMULA:
                val = String.valueOf(cell.getNumericCellValue());
                break;
            case BLANK:
                val = "";
                break;
            case BOOLEAN:
                val = cell.getStringCellValue();
                break;
            case ERROR:
                val = "";
                break;
            default:
                break;
        }
        return val;
    }


}
